<?php

include_spip('inc/documents');
if (!defined('_FORMIDABLE_TAILLE_MAX_FICHIERS_EMAIL')) {// la taille maximum (en Mio) des fichiers qu'on autorise directement en PJ dans les emails.
	define('_FORMIDABLE_TAILLE_MAX_FICHIERS_EMAIL', 10);
}


// Lieux de stockages des fichiers
if (!defined('_DIR_FICHIERS')) { // En attendant que ce soit natif spip
	define('_DIR_FICHIERS', _DIR_ETC . 'fichiers/');
}

if (!defined('_DIR_FICHIERS_FORMIDABLE')) {
	define('_DIR_FICHIERS_FORMIDABLE', _DIR_FICHIERS . 'formidable/');
}

if (!defined('_FORMIDABLE_EXPIRATION_FICHIERS_EMAIL')) {
	// Combien de temps un lien par email dans fichier est valable (en seconde)
	define('_FORMIDABLE_EXPIRATION_FICHIERS_EMAIL', 24 * 3600);
}
if (!defined('_FORMIDABLE_EFFACEMENT_FICHIERS_EMAIL')) {
	// Au bout de combien de temps efface-t-on les fichiers enregistrés lorsque le traitement est uniquement email?
	define('_FORMIDABLE_EFFACEMENT_FICHIERS_EMAIL', _FORMIDABLE_EXPIRATION_FICHIERS_EMAIL);
}
if (!defined('_FORMIDABLE_LIENS_FICHIERS_ACCUSE_RECEPTION')) {
	// mettre à false si on ne veut pas de lien vers les fichiers dans l'accusé de réception
	define('_FORMIDABLE_LIENS_FICHIERS_ACCUSE_RECEPTION', true);
}
/**
 * Créer, si le formulaire contient des saisies de type fichiers, un dossier pour stocker les fichiers.
 * Vérifier que ce dossier soit accessible en écriture.
 * Vérifier qu'on ne puisse pas y accéder de l'exterieur.
 *
 * @param int $id_formulaire
 * @param bool $forcer, pour forcer la création du dossier même si pas de saisie fichiers
 * @return string $erreur
 **/
function formidable_creer_dossier_formulaire($id_formulaire, $forcer = false) {
	if (!$forcer) {
		include_spip('formulaires/formidable');
		// Récuperer la liste des saisies de type fichier
		$saisies_fichiers = formulaires_formidable_fichiers($id_formulaire);

		if (!is_array($saisies_fichiers) || $saisies_fichiers == []) {
			//pas de saisie fichiers?
			return '';
		}
	}
	$nom_dossier = "formulaire_$id_formulaire";

	// On crée le dossier
	sous_repertoire(_DIR_FICHIERS, '', true, true);
	sous_repertoire(_DIR_FICHIERS_FORMIDABLE, '', true, true);
	$dossier = sous_repertoire(_DIR_FICHIERS_FORMIDABLE, $nom_dossier, false, true);
	if (strpos($dossier, "$nom_dossier/") === false) {
		spip_log("Impossible d'écrire $nom_dossier", 'formidable' . _LOG_ERREUR);
		return _T(
			'formidable:creer_dossier_formulaire_erreur_impossible_creer',
			['dossier' => _DIR_FICHIERS_FORMIDABLE . $nom_dossier]
		);
	}

	// Créer un htaccess ici
	include_spip('inc/acces');
	verifier_htaccess($dossier);

	// on crée un fichier de test, pour s'assurer
	// 1. Qu'on puisse écrire dans le rep
	// 2. Qu'on ne puisse pas accéder à ce fichier depuis l'exterieur.
	$fichier = $dossier . 'test.txt';
	$ecriture_ok = ecrire_fichier(
		$fichier,
		"Ce fichier n'est normalement pas lisible de l'extérieur. Si tel est le cas, il y a un souci de confidentialité.",
		false
	);
	if ($ecriture_ok == false) {
		spip_log("Impossible d'écrire dans $nom_dossier", 'formidable' . _LOG_ERREUR);
		return _T(
			'formidable:creer_dossier_formulaire_erreur_impossible_ecrire',
			['dossier' => _DIR_FICHIERS_FORMIDABLE . $nom_dossier]
		);
	}

	include_spip('inc/distant');
	$url = url_absolue($fichier);
	$data = recuperer_url($url, [
		'methode' => 'HEAD'
	]);
	if (($data['status'] ?? '') == 200) {
		// si on peut récuperer la page avec un statut http 200, c'est qu'il y a un problème
		spip_log("$nom_dossier accessible en lecture depuis le web", 'formidable' . _LOG_CRITIQUE);
		return _T(
			'formidable:creer_dossier_formulaire_erreur_possible_lire_exterieur',
			['dossier' => _DIR_FICHIERS_FORMIDABLE . $nom_dossier]
		);
	}
	// Et si tout va bien
	spip_log("Création du dossier $nom_dossier", 'formidable');
	return '';
}

/**
 * Déplace un fichier uploadé de son adresse temporaire vers son adresse définitive.
 * Crée si besoin les dossiers de stockage.
 *
 * @param string $fichier l'adresse temporaire du fichier
 * @param string $nom le nom du fichier
 * @param string $mime le mime du fichier
 * @param string $extension l'extension du fichier
 * @param string $champ le champ concerné
 * @param array $options tableau d'option
 *	- `'id_formulaire'` => int
 *	- `'id_formulaires_reponse'` => int
 *	- `'timestamp'` => int
 * @return string $nom_definitif
 * 		le nom définitif du fichier tel que stocké dans son dossier,
 * 		vide s'il y a eu un souci lors du déplacement (dans ce cas un courriel sera envoyé au webmestre)
 *
 **/
function formidable_deplacer_fichier_emplacement_definitif($fichier, $nom, $mime, $extension, $champ, $options) {
	$nom = formidable_standardiser_nom_fichier($nom);
	if (isset($options['id_formulaire'])) {
		$id_formulaire = $options['id_formulaire'];
		$dossier_formulaire =  "formulaire_$id_formulaire";
	} else {
		// si c'est pas set, c'est qu'il y a une erreur
		return '';
	}

	if (isset($options['id_formulaires_reponse'])) {
		$dossier_reponse = 'reponse_' . $options['id_formulaires_reponse'];
	} elseif (isset($options['timestamp'])) {
		$dossier_reponse = 'reponse_' . $options['timestamp'];
	} else { // si ni timestamp, ni id_formulaires_reponse => erreur
		return '';
	}
	// déterminer le basename
	$basename = pathinfo($nom, PATHINFO_BASENAME);

	// Appliquer les alias de type_mime
	include_spip('base/typedoc');
	while (isset($GLOBALS['mime_alias'][$mime])) {
		$mime = $GLOBALS['mime_alias'][$mime];
	}

	// sécurité : si la combinaison extension/mime_type est inconnu de SPIP (spip_documents_type), on zip.
	// On n'utilise volontairement pas verifier/fichiers.php, dès fois que celui-ci évolue dans le future
	$mimeOK = false;
	$mime_extension = sql_getfetsel('mime_type', 'spip_types_documents', 'extension=' . sql_quote($extension));
	if ($mime_extension) {
		if ($mime_extension === $mime) {
			$mimeOK = true;
		} else {
			// cas particulier des fichiers de type text/... qui risquent à tout moment d'arriver en text/plain
			// car les fichiers text/... n'ont ni structure ni meta-données et la detection du mime n'est pas robuste
			if (str_starts_with($mime_extension, 'text/') && $mime === 'text/plain') {
				$mimeOK = true;
			}
		}
	}
	if (!$mimeOK) {
		$zipper = true;
		$nom_dans_zip = $nom;
		// pas de fichier nom de zip commencant par point
		while (strpos($basename, '.') === 0) {
			$basename = substr($basename, 1);
		}
		$nom = "$basename.zip";
	} else {
		$zipper = false;
	}
	if (!isset($options['timestamp'])) { // si on enregistre la réponse en base
		// d'abord, créer si besoin le dossier pour le formulaire, si on a une erreur, on ne déplace pas le fichier
		if (formidable_creer_dossier_formulaire($id_formulaire, true) != '') {
			return '';
		}

		// puis on créer le dossier pour la réponse
		$dossier_reponse = sous_repertoire(
			_DIR_FICHIERS_FORMIDABLE . $dossier_formulaire . '/',
			$dossier_reponse,
			false,
			true
		);

		// puis le dossier pour le champ
		$dossier_champ = sous_repertoire($dossier_reponse, $champ, false, true);
		$appendice_nom = 0;
	} else { // si on enregistre sous forme de timestamp
		sous_repertoire(_DIR_FICHIERS, '', true, true);
		sous_repertoire(_DIR_FICHIERS_FORMIDABLE, '', true, true);
		$dossier = sous_repertoire(_DIR_FICHIERS_FORMIDABLE, 'timestamp', false, true);
		$dossier = sous_repertoire($dossier, $options['timestamp'], false, true);
		$dossier_champ = sous_repertoire($dossier, $champ, false, true);

		// Générer un fichier htaccess ici
		include_spip('inc/acces');
		verifier_htaccess($dossier);

		// on crée un fichier de test, pour s'assurer
		// 1. Qu'on puisse écrire dans le rep
		// 2. Qu'on ne puisse pas accéder à ce fichier depuis l'exterieur.
		$fichier_test = $dossier . 'test.txt';
		$ecriture_ok = ecrire_fichier(
			$fichier_test,
			"Ce fichier n'est normalement pas lisible de l'extérieur. Si tel est le cas, il y a un souci de confidentialité.",
			false
		);
		if ($ecriture_ok == false) {
			spip_log("Impossible d'écrire dans $dossier", 'formidable' . _LOG_ERREUR);
			return '';
		}
		// WARNING : ce test n'est pas fiable car il repose sur le fait que le serveur est capable d'acceder a ses propres URLs
		// ce qui n'est pas forcement vrai selon la config reseau etc. Peut provoquer un delai d'attente important si le firewall bloque la requete http
		include_spip('inc/distant');
		$url = url_absolue($fichier_test);
		$data = recuperer_url($url, [
			'methode' => 'HEAD'
		]);
		if (($data['status'] ?? '') === 200) {
			// si on peut récuperer la page avec un statut http 200,
			// c'est qu'il y a un problème.
			spip_log("$dossier accessible en lecture depuis le web", 'formidable' . _LOG_CRITIQUE);
			return '';
		}
	}
	// S'assurer qu'il n'y a pas un fichier du même nom à destination
	$chemin_final = $dossier_champ . $nom;
	$n = 1;
	$basename_sans_extension = substr_replace($basename, '', -strlen($extension) - 1);
	while (@file_exists($chemin_final)) {
		$nom = $basename_sans_extension . "_$n." . $extension;
		$chemin_final = $dossier_champ . $nom;
		$n++;
	}
	if (!$zipper) { // si on ne zippe pas, c'est simple
		if ($fichier_dest = deplacer_fichier_upload($fichier, $chemin_final)) {// On se contente de copier le fichier. COmme cela en cas de souci quelconque, c'est conservé au moins dans tmp/cvt_upload. Celui ci a son propre mécanisme de nettoyage.
			spip_log("Enregistrement du fichier $chemin_final", 'formidable');
			return basename($fichier_dest);
		} else {
			spip_log("Pb lors de l'enregistrement du fichier $chemin_final", 'formidable' . _LOG_ERREUR);
			return '';
		}
	} else { // si on doit zipper, c'est plus complexe
		include_spip('inc/archives');
		$zip = new  Spip\Archives\SpipArchives($chemin_final);
		// mettre à jour le fichier dans le dossier cvtupload
		if (!$tmp_dir = tempnam($dossier_champ, 'tmp_upload')) {
			return '';
		}
		spip_unlink($tmp_dir);
		@mkdir($tmp_dir);
		if (!$fichier_tmp = deplacer_fichier_upload($fichier, $tmp_dir . '/' . $nom_dans_zip, false)) {
			spip_log("Pb lors de l'enregistrement du fichier $tmp_dir/$nom_dans_zip", 'formidable' . _LOG_ERREUR);
			return '';
		}
		if (!$zip->emballer([$fichier_tmp], $tmp_dir)) {
			spip_log("Pb lors de l'enregistrement du fichier $fichier", 'formidable' . _LOG_ERREUR);
			return '';
		} else {
			//spip_unlink($fichier); On n'efface pas, car cela permet en cas de souci par ailleurs d'avoir au moins le fichier dans tmp/cvt_upload. CVT upload a son propre mécanisme de nettoyage de ce dossier.
			supprimer_repertoire($tmp_dir);
			spip_log("Enregistrement du fichier $fichier, automatiquement zippé", 'formidable');
			return $nom;
		}
	}

	return $nom;
}

/**
 * Fournit à l'utilisateur·trice un fichier qui se trouve normalement dans un endroit inaccessible,
 * par exemple dans config.
 *
 * La fonction ne vérifie ni l'existence effective du fichier,
 * ni le droit effectif de l'utilisateur.
 * Ceci doit être fait dans l'action qui appelle cette fonction
 *
 * @param string $chemin le chemin du fichier
 * @param string $f le nom du fichier qui sera envoyé à l'utilisateur·trice.
 * @param string $content_type Si Connu, sinon calculé
 *
**/
function formidable_retourner_fichier($chemin, $f, $content_type = '') {
	include_spip('inc/livrer_fichier');
	spip_livrer_fichier($chemin, $content_type, ['attachment' => $f]);
}
/**
 * Déplacer un fichier temporaire à son emplacement définif.
 * Produire un tableau de description des fichiers déplacés.
 * Le tout à partir de la description d'une saisies 'fichiers'
 * @param array $saisie la description de la saisie fichiers
 * @param array $options
 * 		des options, dépendantes du type de traitement,
 * 		qui permettent d'indiquer où l'on déplace le fichier
 * @return array|null un tableau de "vue" de la saisie
**/
function formidable_deplacer_fichiers_produire_vue_saisie($saisie, $options) {
	$nb_fichiers_max = $saisie['options']['nb_fichiers'];
	// on va parcourir $_FILES en nous limitant aux nombres de fichiers définies par la saisie,
	// pour éviter les éventuelles ajout supplémentaire de fichiers par modif du html
	$champ = $saisie['options']['nom'];
	if (!isset($_FILES[$champ])) {//précaution
		return null;
	}
	$description_fichiers = [];
	$mon_file = $_FILES[$champ];
	$i = 0;
	while ($i < $nb_fichiers_max) {
		if (!isset($mon_file['error'][$i])) {
			// Il n'y a pas de tentative d'envoi de fichier pour ce numéro -> on passe
			$i++;
			continue;
		}
		if ($mon_file['error'][$i] === 0) {
			// la saisie fichiers est forcément structurée sous la forme d'un tableau,
			// on peut donc vérifier qu'il n'y a pas d'erreur facilement
			$description = []; // tableau pour stocker la description de ce fichier

			// les infos qu'on peut récuperer directement de $files
			$description['taille'] = $mon_file['size'][$i];
			$description['mime'] = $mon_file['type'][$i];

			// l'adresse du nouveau fichier, sans le chemin
			if (
				$nouveau_nom = formidable_deplacer_fichier_emplacement_definitif(
					$mon_file['tmp_name'][$i],
					$mon_file['name'][$i],
					$mon_file['type'][$i],
					pathinfo($mon_file['name'][$i], PATHINFO_EXTENSION),
					$champ,
					$options
				)
			) {
					$description['nom'] = $nouveau_nom;
					$description['extension'] = pathinfo($nouveau_nom, PATHINFO_EXTENSION);
			} else {
				$description['erreur'] = _T(
					'formidable:erreur_deplacement_fichier',
					['nom' => $mon_file['name'][$i]]
				);
				$description['nom'] = $mon_file['name'][$i];
				$description['tmp_name'] = $mon_file['tmp_name'][$i];
			}
			//on ajoute la description au tableau global
			$description_fichiers[$i] = $description;
		}
		$i++;
	}
	return $description_fichiers;
}
/**
 * Efface les fichiers d'un formulaire
 * @param string $id_formulaire
 * @return int 1 ou 0 selon que l'on a effacé ou non un répertoire
**/
function formidable_effacer_fichiers_formulaire($id_formulaire) {
	$chemin = _DIR_FICHIERS_FORMIDABLE . "formulaire_$id_formulaire";
	if (file_exists($chemin)) {// par sécurité
		if (supprimer_repertoire($chemin)) {
			spip_log("Effacement du dossier $chemin", 'formidable');
			return 1;
		} else {
			spip_log("Pb lors de l'effacement du dossier $chemin", 'formidable' . _LOG_ERREUR);
			return 0;
		}
	}
	return 0;
}

/**
 * Efface les fichiers d'une réponse formidable
 * @param string $id_formulaire
 * @param string $id_formulaires_reponse
 * @return int 1 ou 0 selon que l'on a effacé ou non un répertoire
**/
function formidable_effacer_fichiers_reponse($id_formulaire, $id_formulaires_reponse) {
	$chemin = _DIR_FICHIERS_FORMIDABLE . "formulaire_$id_formulaire/reponse_$id_formulaires_reponse";
	if (file_exists($chemin)) {// par sécurité
		if (supprimer_repertoire($chemin)) {
			spip_log("Effacement du dossier $chemin", 'formidable');
			return 1;
		} else {
			spip_log("Pb lors de l'effacement du dossier $chemin", 'formidable' . _LOG_ERREUR);
			return 0;
		}
	}
	return 0;
}

/** Efface les fichiers d'un champ pour les réponses d'un formulaire
 * @param string $id_formulaire
 * @param array|string $reponses
 * @param string $champ
**/
function formidable_effacer_fichiers_champ($id_formulaire, $reponses, $champ) {
	if ($champ != '') { // on devrait pas arriver ici avec un $champ vide, mais prenons nos précaution
		if (!is_array($reponses)) {
			$reponses = [$reponses];
		}

		$rep_vide = ['.', '..', '.ok']; // si scandire retourne cela où inférieur, alors le dossier est vide
		foreach ($reponses as $rep) {
			$chemin_reponse = _DIR_FICHIERS_FORMIDABLE . "formulaire_$id_formulaire/reponse_$rep";
			$chemin_champ = $chemin_reponse . '/' . $champ;

			if (file_exists($chemin_champ)) {
				if (supprimer_repertoire($chemin_champ)) {
					spip_log("Effacement du dossier $chemin_champ", 'formidable');
				} else {
					spip_log("Pb lors de l'effacement du dossier $chemin_champ", 'formidable' . _LOG_ERREUR);
				}
				if (count(array_diff(scandir($chemin_reponse), $rep_vide)) == 0) {
					// si jamais il ne reste plus aucun fichiers pour cette réponse,
					// on peut effacer le repertoire de celle-ci
					if (supprimer_repertoire($chemin_reponse)) {
						spip_log("Effacement du dossier $chemin_reponse", 'formidable');
					} else {
						spip_log("Pb lors de l'effacement du dossier $chemin_reponse", 'formidable' . _LOG_ERREUR);
					}
				}
			}
		}
	}
}

/** Efface les fichiers des réponses par email
 * lorsque la constante _FORMIDABLE_EFFACEMENT_FICHIERS_EMAIL est différent de 0 et que le temps est écoulé
 * @return int nombre de dossiers effacés
 **/
function formidable_effacer_fichiers_email() {
	if (_FORMIDABLE_EFFACEMENT_FICHIERS_EMAIL == 0) {
		return 0;
	}
	$dossiers_effaces = 0;
	$chemin = _DIR_FICHIERS_FORMIDABLE . 'timestamp';
	$timestamp = time();
	if (is_dir($chemin)) {
		$dossiers = scandir($chemin);
		if (is_array($dossiers)) {
			foreach ($dossiers as $dossier) {
				if (strval(intval($dossier)) != $dossier) { // on ne traite que les dossiers qui ont comme nom un entier
					continue;
				}
				if ($timestamp - intval($dossier) >= _FORMIDABLE_EFFACEMENT_FICHIERS_EMAIL) {
					$chemin_complet = "$chemin/$dossier";
					if (supprimer_repertoire($chemin_complet)) {
						spip_log("Effacement du dossier $chemin_complet", 'formidable');
						$dossiers_effaces++;
					} else {
						spip_log("Pb lors de l'effacement du dossier $chemin_complet", 'formidable' . _LOG_ERREUR);
					}
				}
			}
		}
	}
	return $dossiers_effaces;
}

/**
 * Génerer un zip des réponses d'un formulaire
 *
 * @param int $id_formulaire  (identifiant numérique)
 * @param string $chemin_du_zip chemin complet du zip
 * @param string $fichier_csv un fichier csv à ajouter, contenant les réponses
 * @return string|int chemin complet du zip ou 0 si erreur lors de la création
**/
function formidable_zipper_reponses_formulaire($id_formulaire, $chemin_du_zip, $fichier_csv, $saisies_fichiers) {
	include_spip('inc/archives');
	supprimer_fichier($chemin_du_zip);// Si un précédent zip traine
	$zip = new Spip\Archives\SpipArchives($chemin_du_zip);
	$chemin_fichiers = _DIR_FICHIERS_FORMIDABLE . 'formulaire_' . $id_formulaire;

	//Déplacer le CSV, car Archiviste ne peut emballer qu'une seule fois les fichiers, et ne prend qu'un chemin
	$basename_csv = pathinfo($fichier_csv)['basename'];
	$fichier_csv = deplacer_fichier_upload($fichier_csv, "$chemin_fichiers/$basename_csv", true);


	if (!$zip->emballer(array_merge($saisies_fichiers, [$fichier_csv]), $chemin_fichiers)) {
		spip_log(
			"Impossible de créer le zip pour l'export des réponses du formulaire $id_formulaire",
			'formidable' . _LOG_ERREUR
		);
		return 0;
	} else {
		supprimer_fichier($fichier_csv);//Nettoyage suite déplacement
		return $chemin_du_zip;
	}
}
/**
 * Générer une url d'action pour la récupération d'un fichier lié à une réponse
 * @param int|string $id_formulaire
 * @param int|string $id_formulaires_reponse
 * @param string $saisie
 * @param string $fichier
 **/
function formidable_generer_url_action_recuperer_fichier($id_formulaire, $id_formulaires_reponse, $saisie, $fichier) {
	$param = [
		'formulaire' => $id_formulaire,
		'reponse' => $id_formulaires_reponse,
		'saisie' => $saisie,
		'fichier' => $fichier
	];

	// Pour les utilisateurs non authentifiés, on se base sur le cookier
	$nom_cookie = formidable_generer_nom_cookie($id_formulaire);
	if (isset($_COOKIE[$nom_cookie])) {
		include_spip('inc/securiser_action');
		$param['cookie'] = sha1($_COOKIE[$nom_cookie] . secret_du_site());
	}

	$param = json_encode($param, JSON_UNESCAPED_SLASHES);
	$securiser_action = charger_fonction('securiser_action', 'inc');
	$url = $securiser_action('formidable_recuperer_fichier', $param, '', false);
	spip_log("recuperer_fichier : pour la $saisie et le fichier $fichier avec les options " . print_r($param, true) . " : l'url est $url", 'formidable' . _LOG_DEBUG);
	return $url;
}

/** Générer une url d'action pour récuperer un fichier à partir d'un lien email
 * @param string $saisie
 * @param string $fichier
 * @param array $options décrivant si on récupère par id de réponse ou par timestamp
 * @return string $url
 *
**/
function formidable_generer_url_action_recuperer_fichier_email($saisie, $fichier, $options) {
	if (isset($options['id_formulaires_reponse'])) {//si reponses enregistrées
		$arg = json_encode([
			'formulaire' => strval($options['id_formulaire']),
			'reponse' => strval($options['id_formulaires_reponse']),
			'fichier' => $fichier,
			'saisie' => $saisie
		]);
	} elseif (isset($options['timestamp'])) {//si par timestamp
		$arg = json_encode([
			'timestamp' => strval($options['timestamp']),
			'fichier' => $fichier,
			'saisie' => $saisie
		]);
	} else {
		return '';
	}
	$arg = base64_encode($arg);
	$pass = secret_du_site();
	$action = 'formidable_recuperer_fichier_par_email';
	$hash = _action_auteur("$action-$arg", 0, $pass, 'secret_de_formidable');
	$url = generer_url_action($action, "arg=$arg&hash=$hash&base64=true", true, true);
	spip_log("recuperer_fichier_par_email : pour la $saisie et le fichier $fichier avec les options " . print_r($options, true) . " le arg est $arg et le  hash $hash. Du coup l'url est $url", 'formidable' . _LOG_DEBUG);
	return $url;
}

/**
 * Générer le chemin d'un fichier d'après les paramètres passés en argument
 * @param array $param, paramètres décrivant le fichiers:
 * formulaire => identifiant numérique du formulaire,
 * timestamp => timestamp de la réponse (si la réponse est uniquement envoyée par courriel)
 * reponse => identifiant numérique de la réponse
 * saisie => nom du champ formidable (type fichier_1)
 * fichier => nom du fichier
 * @return string $chemin;
**/
function formidable_generer_chemin_fichier($param) {
	$chemin_fichier = '';
	if (isset($param['reponse'])) {
		$chemin_fichier = _DIR_FICHIERS_FORMIDABLE
			. 'formulaire_' . $param['formulaire']
			. '/reponse_' . $param['reponse']
			. '/' . $param['saisie']
			. '/' . $param['fichier'];
	} elseif (isset($param['timestamp'])) {
		$chemin_fichier = _DIR_FICHIERS_FORMIDABLE
			. 'timestamp/'
			. $param['timestamp'] . '/'
			. $param['saisie'] . '/'
			. $param['fichier'];
	} else {
		include_spip('inc/minipres');
		echo minipres(_T('formidable:erreur_fichier_introuvable'));
	}
	return $chemin_fichier;
}

/**
 * Standardise le nom d'un fichier pour éviter les caractères spéciaux
 * @param string $nom
 * @return string $nom
**/
function formidable_standardiser_nom_fichier(string $nom): string {
	include_spip('inc/charsets');
	$nom = strtolower(translitteration($nom));
	$nom = preg_replace('/<[^>]*>/', '', $nom);
	$nom = preg_replace('/[^.=\w-]+/', '_', $nom);
	return $nom;
}


/**
 * Calcule la taille totale des fichiers
 * d'après une saisie de type fichiers
 * @param array $info_files
 * @return int $taille (en octets)
**/
function formidable_calculer_taille_fichiers_saisie($info_files) {
	return array_sum(array_column($info_files, 'taille'));
}
